// server/index.ts
import express2 from "express";

// server/routes.ts
import { createServer } from "http";

// server/config/db.ts
import mysql from "mysql2/promise";
import dotenv from "dotenv";
dotenv.config();
var dbConfig = {
  host: process.env.MYSQL_HOST || "localhost",
  port: parseInt(process.env.MYSQL_PORT || "3306"),
  user: process.env.MYSQL_USER || "root",
  password: process.env.MYSQL_PASSWORD || "root1234",
  database: process.env.MYSQL_DATABASE || "exam_app",
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};
var pool = mysql.createPool(dbConfig);

// server/database-storage.ts
import session from "express-session";
import connectMysql from "express-mysql-session";
import mysql2 from "mysql2/promise";
var MySQLSessionStore = connectMysql(session);
var DatabaseStorage = class {
  pool;
  sessionStore;
  constructor() {
    const dbConfig2 = {
      host: process.env.MYSQL_HOST || "localhost",
      port: parseInt(process.env.MYSQL_PORT || "3306"),
      user: process.env.MYSQL_USER || "root",
      password: process.env.MYSQL_PASSWORD || "",
      database: process.env.MYSQL_DATABASE || "exam_app",
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0
    };
    this.pool = mysql2.createPool(dbConfig2);
    const sessionOptions = {
      host: dbConfig2.host,
      port: dbConfig2.port,
      user: dbConfig2.user,
      password: dbConfig2.password,
      database: dbConfig2.database,
      createDatabaseTable: true
    };
    this.sessionStore = new MySQLSessionStore(sessionOptions);
  }
  // User methods
  async getUser(id) {
    const [rows] = await pool.query(
      "SELECT * FROM users WHERE id = ?",
      [id]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async getUserByUsername(username) {
    const [rows] = await pool.query(
      "SELECT * FROM users WHERE username = ?",
      [username]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async getUserByEmail(email) {
    const [rows] = await pool.query(
      "SELECT * FROM users WHERE email = ?",
      [email]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async createUser(userData) {
    const [result] = await pool.query(
      "INSERT INTO users (name, email, password_hash, is_admin) VALUES (?, ?, ?, ?)",
      [userData.name, userData.email, userData.password_hash, userData.is_admin || false]
    );
    const user = await this.getUser(result.insertId);
    if (!user) {
      throw new Error("Failed to create user");
    }
    return user;
  }
  async updateUser(id, userData) {
    const fields = [];
    const values = [];
    Object.entries(userData).forEach(([key, value]) => {
      if (key !== "id" && key !== "created_at") {
        fields.push(`${key} = ?`);
        values.push(value);
      }
    });
    if (fields.length === 0) {
      return this.getUser(id);
    }
    values.push(id);
    await pool.query(
      `UPDATE users SET ${fields.join(", ")} WHERE id = ?`,
      values
    );
    return this.getUser(id);
  }
  // Question methods
  async getQuestions(limit, offset = 0) {
    let query = "SELECT * FROM questions";
    const params = [];
    if (limit !== void 0) {
      query += " LIMIT ?";
      params.push(limit);
      if (offset > 0) {
        query += " OFFSET ?";
        params.push(offset);
      }
    }
    const [rows] = await pool.query(query, params);
    return rows;
  }
  async getQuestionsByFilter(topic, difficulty, limit, offset = 0) {
    let query = "SELECT * FROM questions";
    const conditions = [];
    const params = [];
    if (topic) {
      conditions.push("topic = ?");
      params.push(topic);
    }
    if (difficulty) {
      conditions.push("difficulty = ?");
      params.push(difficulty);
    }
    if (conditions.length > 0) {
      query += " WHERE " + conditions.join(" AND ");
    }
    if (limit !== void 0) {
      query += " LIMIT ?";
      params.push(limit);
      if (offset > 0) {
        query += " OFFSET ?";
        params.push(offset);
      }
    }
    const [rows] = await pool.query(query, params);
    return rows;
  }
  async getQuestion(id) {
    const [rows] = await pool.query(
      "SELECT * FROM questions WHERE id = ?",
      [id]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async createQuestion(questionData) {
    const [result] = await pool.query(
      "INSERT INTO questions (title, option_a, option_b, option_c, option_d, correct_answer, explanation, topic, difficulty) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)",
      [
        questionData.title,
        questionData.option_a,
        questionData.option_b,
        questionData.option_c,
        questionData.option_d,
        questionData.correct_answer,
        questionData.explanation,
        questionData.topic,
        questionData.difficulty
      ]
    );
    const question = await this.getQuestion(result.insertId);
    if (!question) {
      throw new Error("Failed to create question");
    }
    return question;
  }
  async updateQuestion(id, questionData) {
    const fields = [];
    const values = [];
    Object.entries(questionData).forEach(([key, value]) => {
      if (key !== "id") {
        fields.push(`${key} = ?`);
        values.push(value);
      }
    });
    if (fields.length === 0) {
      return this.getQuestion(id);
    }
    values.push(id);
    await pool.query(
      `UPDATE questions SET ${fields.join(", ")} WHERE id = ?`,
      values
    );
    return this.getQuestion(id);
  }
  async deleteQuestion(id) {
    const [result] = await pool.query(
      "DELETE FROM questions WHERE id = ?",
      [id]
    );
    return result.affectedRows > 0;
  }
  async countQuestions(topic, difficulty) {
    let query = "SELECT COUNT(*) as count FROM questions";
    const params = [];
    const conditions = [];
    if (topic) {
      conditions.push("topic = ?");
      params.push(topic);
    }
    if (difficulty) {
      conditions.push("difficulty = ?");
      params.push(difficulty);
    }
    if (conditions.length > 0) {
      query += " WHERE " + conditions.join(" AND ");
    }
    const [rows] = await pool.query(query, params);
    return rows[0].count;
  }
  // User Answer methods
  async createUserAnswer(userAnswerData) {
    const [result] = await pool.query(
      "INSERT INTO user_answers (user_id, question_id, selected_answer, is_correct) VALUES (?, ?, ?, ?)",
      [
        userAnswerData.user_id,
        userAnswerData.question_id,
        userAnswerData.selected_answer,
        userAnswerData.is_correct
      ]
    );
    const [rows] = await pool.query(
      "SELECT * FROM user_answers WHERE id = ?",
      [result.insertId]
    );
    if (rows.length === 0) {
      throw new Error("Failed to create user answer");
    }
    return rows[0];
  }
  async getUserAnswers(userId) {
    const [rows] = await pool.query(
      "SELECT * FROM user_answers WHERE user_id = ? ORDER BY answered_at DESC",
      [userId]
    );
    return rows;
  }
  async getRecentUserAnswers(userId, limit) {
    const [rows] = await pool.query(
      "SELECT * FROM user_answers WHERE user_id = ? ORDER BY answered_at DESC LIMIT ?",
      [userId, limit]
    );
    return rows;
  }
  // Subscription methods
  async getSubscription(userId) {
    const [rows] = await pool.query(
      "SELECT * FROM subscriptions WHERE user_id = ?",
      [userId]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async createSubscription(subscriptionData) {
    const [result] = await pool.query(
      "INSERT INTO subscriptions (user_id, plan_type, start_date, end_date, status) VALUES (?, ?, ?, ?, ?)",
      [
        subscriptionData.user_id,
        subscriptionData.plan_type,
        subscriptionData.start_date,
        subscriptionData.end_date,
        subscriptionData.status
      ]
    );
    const [rows] = await pool.query(
      "SELECT * FROM subscriptions WHERE id = ?",
      [result.insertId]
    );
    if (rows.length === 0) {
      throw new Error("Failed to create subscription");
    }
    return rows[0];
  }
  async updateSubscription(id, subscriptionData) {
    const fields = [];
    const values = [];
    Object.entries(subscriptionData).forEach(([key, value]) => {
      if (key !== "id") {
        fields.push(`${key} = ?`);
        values.push(value);
      }
    });
    if (fields.length === 0) {
      const [rows2] = await pool.query(
        "SELECT * FROM subscriptions WHERE id = ?",
        [id]
      );
      if (rows2.length === 0) {
        return void 0;
      }
      return rows2[0];
    }
    values.push(id);
    await pool.query(
      `UPDATE subscriptions SET ${fields.join(", ")} WHERE id = ?`,
      values
    );
    const [rows] = await pool.query(
      "SELECT * FROM subscriptions WHERE id = ?",
      [id]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  // Manual Payment methods
  async createManualPayment(manualPaymentData) {
    const [result] = await pool.query(
      "INSERT INTO manual_payments (user_id, uploaded_file_path, plan_type, status) VALUES (?, ?, ?, ?)",
      [
        manualPaymentData.user_id,
        manualPaymentData.uploaded_file_path,
        manualPaymentData.plan_type,
        manualPaymentData.status || "pending"
      ]
    );
    const [rows] = await pool.query(
      "SELECT * FROM manual_payments WHERE id = ?",
      [result.insertId]
    );
    if (rows.length === 0) {
      throw new Error("Failed to create manual payment");
    }
    return rows[0];
  }
  async getManualPayments(userId) {
    let query = "SELECT * FROM manual_payments";
    const params = [];
    if (userId !== void 0) {
      query += " WHERE user_id = ?";
      params.push(userId);
    }
    query += " ORDER BY timestamp DESC";
    const [rows] = await pool.query(query, params);
    return rows;
  }
  async updateManualPayment(id, status) {
    await pool.query(
      "UPDATE manual_payments SET status = ? WHERE id = ?",
      [status, id]
    );
    const [rows] = await pool.query(
      "SELECT * FROM manual_payments WHERE id = ?",
      [id]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  // Question Note methods
  async getQuestionNotes(userId, questionId) {
    let query = "SELECT * FROM question_notes WHERE user_id = ?";
    const params = [userId];
    if (questionId !== void 0) {
      query += " AND question_id = ?";
      params.push(questionId);
    }
    const [rows] = await pool.query(query, params);
    return rows;
  }
  async createQuestionNote(questionNoteData) {
    const [existingResult] = await pool.query(
      "UPDATE question_notes SET note_text = ? WHERE user_id = ? AND question_id = ?",
      [questionNoteData.note_text, questionNoteData.user_id, questionNoteData.question_id]
    );
    if (existingResult.affectedRows === 0) {
      await pool.query(
        "INSERT INTO question_notes (user_id, question_id, note_text) VALUES (?, ?, ?)",
        [questionNoteData.user_id, questionNoteData.question_id, questionNoteData.note_text]
      );
    }
    const [rows] = await pool.query(
      "SELECT * FROM question_notes WHERE user_id = ? AND question_id = ?",
      [questionNoteData.user_id, questionNoteData.question_id]
    );
    if (rows.length === 0) {
      throw new Error("Failed to create question note");
    }
    return rows[0];
  }
  async updateQuestionNote(id, noteText) {
    await pool.query(
      "UPDATE question_notes SET note_text = ? WHERE id = ?",
      [noteText, id]
    );
    const [rows] = await pool.query(
      "SELECT * FROM question_notes WHERE id = ?",
      [id]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async deleteQuestionNote(id) {
    const [result] = await pool.query(
      "DELETE FROM question_notes WHERE id = ?",
      [id]
    );
    return result.affectedRows > 0;
  }
  // Marked Review methods
  async getMarkedReviews(userId) {
    const [rows] = await pool.query(
      "SELECT * FROM marked_reviews WHERE user_id = ?",
      [userId]
    );
    return rows;
  }
  async createMarkedReview(markedReviewData) {
    const [existingRows] = await pool.query(
      "SELECT id FROM marked_reviews WHERE user_id = ? AND question_id = ?",
      [markedReviewData.user_id, markedReviewData.question_id]
    );
    if (existingRows.length > 0) {
      const [rows2] = await pool.query(
        "SELECT * FROM marked_reviews WHERE id = ?",
        [existingRows[0].id]
      );
      return rows2[0];
    }
    const [result] = await pool.query(
      "INSERT INTO marked_reviews (user_id, question_id) VALUES (?, ?)",
      [markedReviewData.user_id, markedReviewData.question_id]
    );
    const [rows] = await pool.query(
      "SELECT * FROM marked_reviews WHERE id = ?",
      [result.insertId]
    );
    if (rows.length === 0) {
      throw new Error("Failed to create marked review");
    }
    return rows[0];
  }
  async deleteMarkedReview(userId, questionId) {
    const [result] = await pool.query(
      "DELETE FROM marked_reviews WHERE user_id = ? AND question_id = ?",
      [userId, questionId]
    );
    return result.affectedRows > 0;
  }
  // Settings methods
  async getSetting(key) {
    const [rows] = await pool.query(
      "SELECT * FROM settings WHERE `key` = ?",
      [key]
    );
    if (rows.length === 0) {
      return void 0;
    }
    return rows[0];
  }
  async updateSetting(key, value) {
    const [updateResult] = await pool.query(
      "UPDATE settings SET value = ? WHERE `key` = ?",
      [value, key]
    );
    if (updateResult.affectedRows === 0) {
      await pool.query(
        "INSERT INTO settings (`key`, value) VALUES (?, ?)",
        [key, value]
      );
    }
    const setting = await this.getSetting(key);
    if (!setting) {
      throw new Error("Failed to update setting");
    }
    return setting;
  }
};

// server/auth.ts
import passport from "passport";
import { Strategy as LocalStrategy } from "passport-local";
import session2 from "express-session";
import { scrypt, randomBytes, timingSafeEqual } from "crypto";
import { promisify } from "util";
var scryptAsync = promisify(scrypt);
async function hashPassword(password) {
  const salt = randomBytes(16).toString("hex");
  const buf = await scryptAsync(password, salt, 64);
  return `${buf.toString("hex")}.${salt}`;
}
async function comparePasswords(supplied, stored) {
  const [hashed, salt] = stored.split(".");
  const hashedBuf = Buffer.from(hashed, "hex");
  const suppliedBuf = await scryptAsync(supplied, salt, 64);
  return timingSafeEqual(hashedBuf, suppliedBuf);
}
function setupAuth(app2) {
  const storage = new DatabaseStorage();
  const sessionSettings = {
    secret: process.env.SESSION_SECRET || "mrcs-prep-hub-secret",
    resave: false,
    saveUninitialized: false,
    store: storage.sessionStore,
    cookie: {
      secure: process.env.NODE_ENV === "production",
      maxAge: 1e3 * 60 * 60 * 24 * 7
      // 1 week
    }
  };
  app2.set("trust proxy", 1);
  app2.use(session2(sessionSettings));
  app2.use(passport.initialize());
  app2.use(passport.session());
  passport.use(
    new LocalStrategy(
      {
        usernameField: "email",
        passwordField: "password_hash"
      },
      async (email, password, done) => {
        try {
          const user = await storage.getUserByEmail(email);
          if (!user || !await comparePasswords(password, user.password_hash)) {
            return done(null, false);
          } else {
            return done(null, user);
          }
        } catch (error) {
          return done(error);
        }
      }
    )
  );
  passport.serializeUser((user, done) => {
    done(null, user.id);
  });
  passport.deserializeUser(async (id, done) => {
    try {
      const user = await storage.getUser(id);
      done(null, user);
    } catch (error) {
      done(error);
    }
  });
  app2.post("/api/register", async (req, res, next) => {
    try {
      const existingUser = await storage.getUserByEmail(req.body.email);
      if (existingUser) {
        return res.status(400).send("Email already registered");
      }
      const user = await storage.createUser({
        ...req.body,
        password_hash: await hashPassword(req.body.password_hash)
      });
      req.login(user, (err) => {
        if (err) return next(err);
        res.status(201).json(user);
      });
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/login", passport.authenticate("local"), (req, res) => {
    res.status(200).json(req.user);
  });
  app2.post("/api/logout", (req, res, next) => {
    req.logout((err) => {
      if (err) return next(err);
      res.sendStatus(200);
    });
  });
  app2.get("/api/user", (req, res) => {
    if (!req.isAuthenticated()) return res.sendStatus(401);
    res.json(req.user);
  });
}

// server/routes.ts
import path from "path";
import fs from "fs";
import multer from "multer";
var uploadDir = path.join(process.cwd(), "uploads");
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
}
var storage2 = multer.diskStorage({
  destination: function(req, file, cb) {
    cb(null, uploadDir);
  },
  filename: function(req, file, cb) {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(null, `${uniqueSuffix}-${file.originalname}`);
  }
});
var upload = multer({
  storage: storage2,
  limits: {
    fileSize: 10 * 1024 * 1024
    // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    const allowedMimes = ["image/jpeg", "image/png", "image/gif", "application/pdf"];
    if (allowedMimes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error("Invalid file type. Only JPEG, PNG, GIF, and PDF are allowed."));
    }
  }
});
function isAuthenticated(req, res, next) {
  if (req.isAuthenticated()) {
    return next();
  }
  res.status(401).json({ message: "Unauthorized" });
}
function isAdmin(req, res, next) {
  if (req.isAuthenticated() && req.user && req.user.is_admin) {
    return next();
  }
  res.status(403).json({ message: "Forbidden" });
}
var db = new DatabaseStorage();
async function registerRoutes(app2) {
  setupAuth(app2);
  app2.get("/api/questions", async (req, res, next) => {
    try {
      const limit = req.query.limit ? parseInt(req.query.limit) : void 0;
      const offset = req.query.offset ? parseInt(req.query.offset) : 0;
      const topic = req.query.topic;
      const difficulty = req.query.difficulty;
      if (!req.isAuthenticated()) {
        const guestLimit = await db.getSetting("guest_question_limit");
        const guestLimitNumber = guestLimit ? parseInt(guestLimit.value) : 10;
        const questions2 = await db.getQuestionsByFilter(
          topic,
          difficulty,
          Math.min(limit || guestLimitNumber, guestLimitNumber),
          offset
        );
        return res.json({
          data: questions2,
          meta: {
            total: guestLimitNumber,
            isGuest: true,
            limit: guestLimitNumber
          }
        });
      }
      const questions = await db.getQuestionsByFilter(topic, difficulty, limit, offset);
      const total = await db.countQuestions(topic, difficulty);
      res.json({
        data: questions,
        meta: {
          total,
          isGuest: false
        }
      });
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/questions/:id", async (req, res, next) => {
    try {
      const questionId = parseInt(req.params.id);
      const question = await db.getQuestion(questionId);
      if (!question) {
        return res.status(404).json({ message: "Question not found" });
      }
      res.json(question);
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/questions", isAdmin, async (req, res, next) => {
    try {
      const question = await db.createQuestion(req.body);
      res.status(201).json(question);
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/questions/:id", isAdmin, async (req, res, next) => {
    try {
      const questionId = parseInt(req.params.id);
      const updatedQuestion = await db.updateQuestion(questionId, req.body);
      if (!updatedQuestion) {
        return res.status(404).json({ message: "Question not found" });
      }
      res.json(updatedQuestion);
    } catch (error) {
      next(error);
    }
  });
  app2.delete("/api/questions/:id", isAdmin, async (req, res, next) => {
    try {
      const questionId = parseInt(req.params.id);
      const result = await db.deleteQuestion(questionId);
      if (!result) {
        return res.status(404).json({ message: "Question not found" });
      }
      res.status(204).send();
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/user-answers", isAuthenticated, async (req, res, next) => {
    try {
      const userAnswer = await db.createUserAnswer({
        ...req.body,
        user_id: req.user.id
      });
      res.status(201).json(userAnswer);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/user-answers", isAuthenticated, async (req, res, next) => {
    try {
      const userAnswers = await db.getUserAnswers(req.user.id);
      res.json(userAnswers);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/performance", async (req, res, next) => {
    try {
      if (!req.isAuthenticated()) {
        const totalQuestions = await db.countQuestions();
        return res.json({
          totalAnswered: 0,
          correctAnswers: 0,
          accuracy: 0,
          topicAccuracy: [],
          markedReviewsCount: 0,
          recentAnswers: [],
          subscription: null,
          isGuest: true,
          totalQuestions,
          guestCorrect: 0,
          // Include a sample topic for guests to see what data would look like
          guestTopics: [
            { topic: "Anatomy", description: "Detailed breakdown available with subscription" },
            { topic: "Physiology", description: "Track your progress across all topics" },
            { topic: "Pathology", description: "Identify your strengths and weaknesses" }
          ]
        });
      }
      const userAnswers = await db.getUserAnswers(req.user.id);
      const recentAnswers = await db.getRecentUserAnswers(req.user.id, 10);
      const totalAnswered = userAnswers.length;
      const correctAnswers = userAnswers.filter((a) => a.is_correct).length;
      const accuracy = totalAnswered > 0 ? Math.round(correctAnswers / totalAnswered * 100) : 0;
      const answersByTopic = {};
      for (const answer of userAnswers) {
        const question = await db.getQuestion(answer.question_id);
        if (question) {
          if (!answersByTopic[question.topic]) {
            answersByTopic[question.topic] = { total: 0, correct: 0 };
          }
          answersByTopic[question.topic].total++;
          if (answer.is_correct) {
            answersByTopic[question.topic].correct++;
          }
        }
      }
      const topicAccuracy = Object.entries(answersByTopic).map(([topic, stats]) => ({
        topic,
        accuracy: Math.round(stats.correct / stats.total * 100),
        total: stats.total
      }));
      const markedReviews = await db.getMarkedReviews(req.user.id);
      const subscription = await db.getSubscription(req.user.id);
      res.json({
        totalAnswered,
        correctAnswers,
        accuracy,
        topicAccuracy,
        markedReviewsCount: markedReviews.length,
        recentAnswers,
        subscription,
        isGuest: false
      });
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/notes", isAuthenticated, async (req, res, next) => {
    try {
      const note = await db.createQuestionNote({
        user_id: req.user.id,
        question_id: req.body.question_id,
        note_text: req.body.note_text
      });
      res.status(201).json(note);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/notes", isAuthenticated, async (req, res, next) => {
    try {
      const questionId = req.query.question_id ? parseInt(req.query.question_id) : void 0;
      const notes = await db.getQuestionNotes(req.user.id, questionId);
      res.json(notes);
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/notes/:id", isAuthenticated, async (req, res, next) => {
    try {
      const noteId = parseInt(req.params.id);
      const updatedNote = await db.updateQuestionNote(noteId, req.body.note_text);
      if (!updatedNote) {
        return res.status(404).json({ message: "Note not found" });
      }
      res.json(updatedNote);
    } catch (error) {
      next(error);
    }
  });
  app2.delete("/api/notes/:id", isAuthenticated, async (req, res, next) => {
    try {
      const noteId = parseInt(req.params.id);
      const result = await db.deleteQuestionNote(noteId);
      if (!result) {
        return res.status(404).json({ message: "Note not found" });
      }
      res.status(204).send();
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/marked-reviews", isAuthenticated, async (req, res, next) => {
    try {
      const markedReview = await db.createMarkedReview({
        user_id: req.user.id,
        question_id: req.body.question_id
      });
      res.status(201).json(markedReview);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/marked-reviews", isAuthenticated, async (req, res, next) => {
    try {
      const markedReviews = await db.getMarkedReviews(req.user.id);
      res.json(markedReviews);
    } catch (error) {
      next(error);
    }
  });
  app2.delete("/api/marked-reviews/:questionId", isAuthenticated, async (req, res, next) => {
    try {
      const questionId = parseInt(req.params.questionId);
      const result = await db.deleteMarkedReview(req.user.id, questionId);
      if (!result) {
        return res.status(404).json({ message: "Marked review not found" });
      }
      res.status(204).send();
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/subscription", isAuthenticated, async (req, res, next) => {
    try {
      const subscription = await db.getSubscription(req.user.id);
      res.json(subscription || null);
    } catch (error) {
      next(error);
    }
  });
  app2.post("/api/payments", isAuthenticated, upload.single("receipt"), async (req, res, next) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "Receipt file is required" });
      }
      const payment = await db.createManualPayment({
        user_id: req.user.id,
        plan_type: req.body.plan_type,
        status: "pending",
        uploaded_file_path: req.file.path
      });
      res.status(201).json(payment);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/payments", isAuthenticated, async (req, res, next) => {
    try {
      if (!req.user.is_admin) {
        const payments2 = await db.getManualPayments(req.user.id);
        return res.json(payments2);
      }
      const payments = await db.getManualPayments();
      res.json(payments);
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/payments/:id/approve", isAdmin, async (req, res, next) => {
    try {
      const paymentId = parseInt(req.params.id);
      const payment = await db.updateManualPayment(paymentId, "approved");
      if (!payment) {
        return res.status(404).json({ message: "Payment not found" });
      }
      const existingSubscription = await db.getSubscription(payment.user_id);
      const now = /* @__PURE__ */ new Date();
      let endDate = /* @__PURE__ */ new Date();
      if (payment.plan_type === "monthly") {
        endDate.setMonth(now.getMonth() + 1);
      } else if (payment.plan_type === "six_months") {
        endDate.setMonth(now.getMonth() + 6);
      } else if (payment.plan_type === "annual") {
        endDate.setFullYear(now.getFullYear() + 1);
      }
      if (existingSubscription) {
        await db.updateSubscription(existingSubscription.id, {
          plan_type: payment.plan_type,
          start_date: now,
          end_date: endDate,
          status: "active"
        });
      } else {
        await db.createSubscription({
          user_id: payment.user_id,
          plan_type: payment.plan_type,
          start_date: now,
          end_date: endDate,
          status: "active"
        });
      }
      res.json(payment);
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/payments/:id/reject", isAdmin, async (req, res, next) => {
    try {
      const paymentId = parseInt(req.params.id);
      const payment = await db.updateManualPayment(paymentId, "rejected");
      if (!payment) {
        return res.status(404).json({ message: "Payment not found" });
      }
      res.json(payment);
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/settings", async (req, res, next) => {
    try {
      const guestLimit = await db.getSetting("guest_question_limit");
      res.json({
        guest_question_limit: guestLimit ? parseInt(guestLimit.value) : 10
      });
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/settings/guest-limit", isAdmin, async (req, res, next) => {
    try {
      const limit = req.body.limit.toString();
      const setting = await db.updateSetting("guest_question_limit", limit);
      res.json({ guest_question_limit: parseInt(setting.value) });
    } catch (error) {
      next(error);
    }
  });
  app2.get("/api/users", isAdmin, async (req, res, next) => {
    try {
      res.status(501).json({ message: "Not implemented in memory storage" });
    } catch (error) {
      next(error);
    }
  });
  app2.put("/api/users/:id/promote", isAdmin, async (req, res, next) => {
    try {
      const userId = parseInt(req.params.id);
      const user = await db.updateUser(userId, { is_admin: true });
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }
      res.json(user);
    } catch (error) {
      next(error);
    }
  });
  const httpServer = createServer(app2);
  return httpServer;
}

import express from "express";
import fs2 from "fs";
import path3 from "path";

import path2 from "path";
import { fileURLToPath } from "url";
var __filename = fileURLToPath(import.meta.url);

function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
function serveStatic(app2) {
  const distPath = path4.resolve(__dirname2, "public");
  if (!fs2.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express2.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path4.resolve(distPath, "index.html"));
  });
}

// server/index.ts
import cors from "cors";
import path4 from "path";
import { fileURLToPath as fileURLToPath2 } from "url";
var app = express2();
app.use(cors({
  origin: "http://localhost:5173",
  credentials: true
}));
app.use(express2.json());
app.use(express2.urlencoded({ extended: false }));
app.use((req, res, next) => {
  const start = Date.now();
  const path5 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path5.startsWith("/api")) {
      let logLine = `${req.method} ${path5} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log(logLine);
    }
  });
  next();
});
var __filename2 = fileURLToPath2(import.meta.url);
var __dirname2 = path4.dirname(__filename2);
app.use(express2.static(path4.join(__dirname2, "public")));
(async () => {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    res.status(status).json({ message });
    throw err;
  });

  // فقط serveStatic بدون أي شرط
  serveStatic(app);

  app.get("*", (_, res) => {
    res.sendFile(path4.join(__dirname2, "public", "index.html"));
  });
  const port = 5e3;
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true
  }, () => {
    log(`serving on port ${port}`);
  });
})();
